<?php
/*
*
* JetBackup @ package
* Created By Idan Ben-Ezra
*
* Copyrights @ JetApps
* http://www.jetapps.com
*
**/
namespace  JetBackup\Ajax\Calls;

use JetBackup\Ajax\ListRecord;
use JetBackup\BackupJob\BackupJob;
use JetBackup\CLI\CLI;
use JetBackup\Data\Engine;
use JetBackup\Entities\Util;
use JetBackup\Exception\AjaxException;
use JetBackup\Exception\DBException;
use JetBackup\Exception\JetBackupLinuxException;
use JetBackup\Factory;
use JetBackup\JetBackup;
use JetBackup\JetBackupLinux\JetBackupLinux;
use JetBackup\Snapshot\Snapshot;
use JetBackup\Snapshot\SnapshotItem;
use JetBackup\SocketAPI\Exception\SocketAPIException;
use JetBackup\UserInput\UserInput;
use JetBackup\Wordpress\Helper;
use SleekDB\Exceptions\InvalidArgumentException;
use SleekDB\Exceptions\IOException;

defined("__JETBACKUP__") or die("Restricted Access.");

class FileManager extends ListRecord {

	/**
	 * @throws AjaxException
	 */
	public function getId():int { return $this->getUserInput(JetBackup::ID_FIELD, 0, UserInput::UINT); }

	/**
	 * @throws AjaxException
	 */
	public function getPath():string { return $this->getUserInput('location', '', UserInput::STRING); }

	/**
	 * @throws DBException
	 * @throws AjaxException
	 * @throws SocketAPIException
	 * @throws IOException
	 * @throws InvalidArgumentException
	 */
	public function execute():void {

		$snap = new Snapshot($this->getId());

		if(!$snap->getId()) throw new AjaxException("Invalid snapshot id provided");
		if($snap->getEngine() != Engine::ENGINE_JB) throw new AjaxException("This feature is only supported for JetBackup Linux snapshots");
		if($snap->getBackupType() != BackupJob::TYPE_ACCOUNT) throw new AjaxException("This feature is only supported for account backups");
		if ($snap->getContains() != BackupJob::BACKUP_ACCOUNT_CONTAINS_HOMEDIR &&
		    $snap->getContains() != BackupJob::BACKUP_ACCOUNT_CONTAINS_FULL) {
			throw new AjaxException("This feature is only supported for backups containing the home directory");
		}

		$item = null;
		$items = $snap->getItems();

		foreach($items as $item_details) {
			if($item_details->getBackupContains() != BackupJob::BACKUP_ACCOUNT_CONTAINS_HOMEDIR) continue;
			$item = $item_details;
			break;
		}

		if(!$item) throw new AjaxException("This backup doesn't contain homedir item");
		if(!$userHomedir = Helper::getUserHomedir())throw new AjaxException("Unable to fetch linux user homedir");

		$wordpressHomedir = Factory::getWPHelper()->getWordPressHomedir();

		if(!str_starts_with($wordpressHomedir, $userHomedir))
			throw new AjaxException("The WP public dir doesn't start with the account homedir");

		$public_dir = substr($wordpressHomedir, strlen($userHomedir)+1);
		
		$path = preg_replace("#/+#", "/", $public_dir . JetBackup::SEP . $this->getPath());

		$sort = $this->getSort();
		if(isset($sort['_id'])) {
			$sort['name'] = $sort['_id'];
			unset($sort['_id']);
		}


		try {
			$response = JetBackupLinux::fileManager($item->getUniqueId(), $path, $this->getLimit(), $this->getSkip(), $sort);
		} catch(JetBackupLinuxException $e) {
			throw new AjaxException($e->getMessage());
		}

		if(!$this->isCLI()) {
			$this->setResponseData($response);
			return;
		}
		
		$output = [];

		foreach ($response['files'] as $file) {
			$output[] = [
				//'ID'        => $file['id'],
				'Name'      => $file['name'] . ($file['link'] ? ' -> ' . $file['link'] : ''),
				'Type'      => $file['icon'] == 'dir' ? 'Directory' : 'File',
				'Created'   => CLI::date(strtotime($file['created'])),
				'Size'      => Util::bytesToHumanReadable($file['size']),
			];
		}

		$this->setResponseData($output);
	}
}