<?php

namespace JetBackup\Wordpress;

use JetBackup\Alert\Alert;
use JetBackup\Crontab\Crontab;
use JetBackup\Destination\Destination;
use JetBackup\Exception\DBException;
use JetBackup\Exception\QueueException;
use JetBackup\Factory;
use JetBackup\JetBackup;
use Plugin_Upgrader;
use SleekDB\Exceptions\InvalidArgumentException;
use SleekDB\Exceptions\IOException;

if (!defined( '__JETBACKUP__')) die('Direct access is not allowed');

class Installer {
	
	private function __construct() {} // only static method

	public static function install():void {
		
		try {

			if (version_compare(PHP_VERSION, JetBackup::MINIMUM_PHP_VERSION, '<'))
				throw new \Exception('Error: PHP version ' . JetBackup::MINIMUM_PHP_VERSION . ' or higher is required, Current version: ' . PHP_VERSION);

			if (version_compare(Wordpress::getVersion(), JetBackup::MINIMUM_WP_VERSION, '<'))
				throw new \Exception('WordPress version ' . JetBackup::MINIMUM_WP_VERSION . ' or higher is required. Current version: ' . Wordpress::getVersion());

			$plugins = Wordpress::getPlugins();
			
			foreach (JetBackup::PLUGIN_CONFLICTS as $plugin) {
				if(!isset($plugins[$plugin]) || !Wordpress::isPluginActive($plugin)) continue;
				throw new \Exception('Conflicting plugin found: Using the ' . $plugins[$plugin]['Name'] . ' plugin alongside with JetBackup might interfere with our plugin functionality and is likely to cause errors, Please disable this plugin before using JetBackup.');
			}
			
		} catch (\Exception $e) {
			wp_die(__('This plugin cannot be activated, Error : ' . $e->getMessage(), 'text-domain'), __('Plugin Activation Error', 'text-domain'), array(
				'back_link' => true,
			));
		}
    }

	public static function uninstall():void {
		$cron = new Crontab();
		$cron->removeCrontab();
	}

	public static function deactivate():void {
		$cron = new Crontab();
		$cron->removeCrontab();
	}

	/**
	 * @throws \JetBackup\Exception\IOException
	 */
	public static function update($upgrader):void {

		if (
			!isset($upgrader->new_plugin_data) ||
			!is_array($upgrader->new_plugin_data) ||
			!isset($upgrader->new_plugin_data['Name']) ||
			$upgrader->new_plugin_data['Name'] != JetBackup::PLUGIN_EXT_NAME
		) {
			return;
		}

		
		$config = Factory::getConfig();
		$current_version = $config->getCurrentVersion();

		if(!$current_version || version_compare($current_version, '3.1', '<')) {

			try {

				// Add all destinations to Queue
				$list = Destination::query()->getQuery()->fetch();

				foreach ($list as $destination_details) {
					$destination = new Destination($destination_details[JetBackup::ID_FIELD]);
					$destination->addToQueue(false);
				}
				Alert::add(
					"JetBackup has been upgraded to version " . JetBackup::VERSION,
					"As part of the upgrade to version " . JetBackup::VERSION . ", all remote destinations have been removed. Please re-add your remote destinations to continue using them.",
					Alert::LEVEL_CRITICAL
				);

			} catch (\Exception $e) {
				Alert::add(
					"JetBackup encountered an error during the update process",
					"Error: " . $e->getMessage(),
					Alert::LEVEL_CRITICAL
				);
				return;
			}

		}

		$config->setCurrentVersion(JetBackup::VERSION);
		$config->save();
	}
}
